from itsdangerous import URLSafeTimedSerializer
from compare import create_compare
import mail_utils
import users
import views
from flask import (
    Flask,
    render_template,
    request,
    redirect,
    url_for,
    flash,jsonify
)
from flask_login import LoginManager, login_required, current_user
from werkzeug.security import check_password_hash, generate_password_hash
from flask_mail import Mail
from flask_migrate import Migrate
from forms import EditCallForm, CreateCallForm, UpdatePasswordForm
from config import Config
from models import db, User, Call
from mail_utils import send_new_transcripts_email
import json
import secrets
from utils import (
    get_environmental_variables,
    recreate_all_tables,
    recreate_call_table,
    get_call_table_size,
    get_user_table_size,
    get_call_by_id,
    get_calls_by_company,
    write_users_to_csv,
    create_initial_users,
)
from sqlalchemy.orm.attributes import flag_modified
from transcripts import check_favorites, manage_transcripts
from summary import create_summary

# Define login_manager globally
login_manager = LoginManager()


def create_app(config_class=Config):
    """Creates and configures the Flask application."""
    app = Flask(__name__)
    app.config.from_object(config_class)

    # Initialize database and migrations
    db.init_app(app)
    migrate = Migrate(app, db)

  # Initialize login manager (within create_app)
    login_manager.init_app(app)
    login_manager.login_view = "login"  # Specify the login route

    # Initialize mail
    mail = Mail(app)

    # Register routes
    app.add_url_rule("/", view_func=views.index)
    app.add_url_rule("/logout", methods=["GET", "POST"], view_func=views.logout)
    app.add_url_rule("/login", methods=["GET", "POST"], view_func=views.login)
    app.add_url_rule("/reset_password", methods=["GET", "POST"], view_func=views.send_password_reset)
    app.add_url_rule("/confirm_email/<token>", view_func=mail_utils.confirm_email)
    app.add_url_rule("/update_password/<token>", methods=["GET", "POST"], view_func=views.process_password_reset)
    app.add_url_rule("/options", methods=["GET", "POST"], view_func=views.options)
    
    ### Add the USER routes to the app ###
    app.add_url_rule("/register", methods=["GET", "POST"], view_func=views.register)
    app.add_url_rule("/resend_verification", methods=["POST","GET"], view_func=views.resend_verification)


    # --- CHANGE THIS LINE ---
    # Original: app.add_url_rule("/create_user_route", methods=["POST"], view_func=login_required(users.create_user_route))
    # New: Point to the function now in views.py
    app.add_url_rule("/create_user_route", methods=["GET", "POST"], view_func=login_required(views.create_user_route)) # Added GET method too if needed for displaying the form
    # --- END OF CHANGE ---

        # --- START OF CHANGE ---
    # Original: app.add_url_rule("/edit_user/<int:user_id>", methods=["GET", "POST"], view_func=login_required(users.edit_user))
    # New: Point to the function now in views.py
    app.add_url_rule("/edit_user/<int:user_id>", methods=["GET", "POST"], view_func=login_required(views.edit_user_route))
    # --- END OF CHANGE ---

    app.add_url_rule("/delete_selected_users", methods=['POST'], view_func=login_required(views.delete_selected_users))
    app.add_url_rule("/delete_user/<int:user_id>", view_func=login_required(views.delete_user))
    app.add_url_rule("/save_users", methods=["POST"], view_func=login_required(users.save_users))
    app.add_url_rule("/load_users", methods=["POST"], view_func=login_required(users.load_users))
    app.add_url_rule("/recreate_user", methods=["POST"], view_func=login_required(users.recreate_user))
    app.add_url_rule("/cleanup_inactive_users_route", methods=["POST"], view_func=login_required(views.cleanup_inactive_users_route))
    # ... (Add other app setup code here if needed) ...
    app.secret_key = secrets.token_hex(16)
    return app


# Create the app instance
app = create_app()
s = URLSafeTimedSerializer(app.secret_key)
env_vars = get_environmental_variables()
SMTP_PASSWORD = env_vars["SMTP_PASSWORD"]
print("SMTP Password =", SMTP_PASSWORD)


@app.route("/recreate_tables", methods=["POST"])
@login_required
def recreate_tables():
    if request.method == "POST":
        write_users_to_csv(db)
        recreate_all_tables(db)
        create_initial_users(db)
        flash("All tables recreated successfully!", "success")
        return redirect(url_for("admin"))


@app.route("/recreate_call", methods=["POST"])
@login_required
def recreate_call():
    if request.method == "POST":
        recreate_call_table(db)
        flash("Call table recreated successfully!", "success")
        return redirect(url_for("admin"))


@app.route("/admin")
@login_required
def admin():
    if current_user.get_id() == "1":
        number_of_calls = get_call_table_size(db)
        number_of_users = get_user_table_size(db)
        return render_template("admin.html", num_calls=number_of_calls, num_users=number_of_users)
    else:
        return redirect(url_for("index"))


@app.route('/change_password', methods=['GET', 'POST'])
@login_required
def change_password():
    form = UpdatePasswordForm()
    if form.validate_on_submit():
        if check_password_hash(current_user.password, form.current_password.data):
            current_user.password = generate_password_hash(form.new_password.data)  # Use scrypt (default)
            db.session.commit()
            flash('Password updated successfully!', 'success')
            return redirect(url_for('dashboard'))
        else:
            flash('Incorrect current password.', 'danger')
    return render_template("dashboard.html", form=form)


@app.route("/delete_favorite/<string:symbol>", methods=["GET"])
@login_required
def delete_favorite(symbol):
    user = User.query.get(current_user.id)
    if user:
        if user.favorites:
            try:
                favorites_list = json.loads(user.favorites)  # Convert to list
            except (TypeError, json.JSONDecodeError):  # Handle cases where user.favorites is None or invalid JSON
                favorites_list = []

            if symbol in favorites_list:
                favorites_list.remove(symbol)
                user.favorites = json.dumps(favorites_list)  # Convert back to JSON string
                flag_modified(user, "favorites") # Required by SQLAlchemy to mark changes of mutable objects.
                db.session.commit()
                flash(f"{symbol} removed from your favorites!", "success")
            else:
                flash(f"{symbol} not found in your favorites.", "info") # Clearer message
        else:
            flash("Your favorites list is empty.", "info") # Handle the empty list case.
    else:
        flash("User not found.", "danger")
    return redirect(url_for('dashboard'))


@app.route('/add_favorite', methods=['POST'])
def add_favorite():
    symbol = request.form.get('symbol')
    if not symbol:
        flash('Ticker symbol is required.', 'danger')
        return redirect(url_for('dashboard'))

    try:
        user = User.query.filter_by(id=current_user.id).first_or_404()

        if user.favorites is None: # Important: Initialize user.favorites. In the DB, also set a default for the favorites field.
            user.favorites = json.dumps([])  # Initialize as empty JSON list/array if new.

        favorites_list = json.loads(user.favorites)  # convert json string to list.

        if symbol.upper() not in favorites_list:
            favorites_list.append(symbol.upper())
            user.favorites = json.dumps(favorites_list) # Convert back to json string. Required by SQLAlchemy to mark changes to JSON field.
            db.session.commit()
            flash(f'Added {symbol.upper()} to your favorites!', 'success')
        else:
            flash(f'{symbol.upper()} already in your favorites.', 'info')

    except Exception as e:
        db.session.rollback()
        app.logger.error(f"Error adding favorite: {e}")
        flash('An error occurred adding the favorite. Please try again later.', 'danger')
    return redirect(url_for('dashboard'))


@app.route("/process_transcripts/<int:number_to_process>/<int:offset>", methods=["POST"])
@login_required
def process_transcripts(number_to_process, offset):
    new_symbols = manage_transcripts(db, number_to_process, offset)
    print(f"New symbols processed: {new_symbols}")
    matching_favorites = check_favorites(db, new_symbols)
    print(f"Matching favorites: {matching_favorites}")
    send_new_transcripts_email(db, matching_favorites)    
    return redirect(url_for("admin"))


@app.route("/search_calls", methods=["GET"])
def search_calls():
    query = request.args.get("query")
    page = request.args.get("page", 1, type=int)
    per_page = 12

    if query:
        query = query.upper().strip()

        pagination = (db.session.query(Call)
            .filter((Call.companyname.startswith(f"{query}%"))
                | (Call.symbol.startswith(f"{query}%")))
            .order_by(
                db.case(
                    (Call.companyname == query, 1), (Call.symbol == query, 1), else_=2),
                Call.date.desc(),  # Sort by date in descending order (most recent first)
                Call.companyname.asc(),
                Call.symbol.asc(),
            )
            .paginate(page=page, per_page=per_page)
        )
        search_results = pagination.items
    else:
        pagination = None
        search_results = []

    return render_template("search_results.html", calls=search_results, query=query, pagination=pagination)


@app.route("/call/<int:call_id>")
def call_page(call_id):
    call = get_call_by_id(db, call_id)
    if call and call.json_transcript:  # Check if call and json_transcript exist
        transcript_data = json.loads(call.json_transcript)
        participants = call.participants
        return render_template(
            "call_page.html",
            call=call,
            participants=participants,
            transcript_data=transcript_data,
        )
    else:
        flash("Transcript not found for this call.", "warning")  # Or similar user-friendly message
        return redirect(url_for('index')) # Redirect to a suitable page


@app.route('/process_selected_calls', methods=['POST'])
def process_selected_calls():
    selected_call_ids = [int(call_id) for call_id in request.form.getlist('selected_calls')]
    print(selected_call_ids)
    text = create_compare(db, selected_call_ids[0], selected_call_ids[1])
    return render_template("compare.html", text=text)


@app.route("/summary_page/<int:call_id>")
def summary_page(call_id):
    call = create_summary(db,call_id)
    return render_template("summary_page.html", call=call)


@app.route("/company/<string:symbol>")
def company_page(symbol):
    calls = get_calls_by_company(db, symbol)
    if not calls:
        # Handle the case where no calls are found
        return "No calls found for this company", 404
    return render_template("company_page.html", symbol=symbol, calls=calls)


@app.route("/delete_call/<int:call_id>", methods=["GET", "POST"])
def delete_call(call_id):
    call = Call.query.get_or_404(call_id)
    if call:
        db.session.delete(call)
        db.session.commit()
        flash("Call deleted", "info")
        flash(f"Call {call_id} has been deleted.", "success")
    else:
        flash(f"Call {call_id} not found.", "danger")
    return redirect(url_for("show_calls"))


@app.route("/create_call", methods=["GET", "POST"])
def create_call():
    form = CreateCallForm()
    new_call = Call()
    if form.validate_on_submit():
        form.populate_obj(new_call)  # Update new_call object with form data
        db.session.add(new_call)
        db.session.commit()
        flash("Call Created", "info")
        return redirect(url_for("show_calls"))
    return render_template("create_call.html", form=form)


@app.route("/edit_call/<int:call_id>", methods=["GET", "POST"])
def edit_call(call_id):
    call = Call.query.get_or_404(call_id)
    form = EditCallForm(obj=call)  # Populate form fields from the call object
    if form.validate_on_submit():
        form.populate_obj(call)  # Update call object with form data
        db.session.commit()
        flash("Call updated", "info")
        return redirect(url_for("show_calls"))

    return render_template("edit_call.html", form=form)


@app.route("/show_calls", methods=["GET", "POST"])
def show_calls():
    page = request.args.get("page", 1, type=int)
    per_page = 25
    # Sort calls by date in descending order (most recent first)
    all_calls = Call.query.order_by(Call.date.desc()).paginate(
        page=page, per_page=per_page, error_out=False
    )
    return render_template("show_calls.html", calls=all_calls.items, pagination=all_calls)


@app.route('/show_users', methods=['GET', 'POST'])
def show_users():
    page = request.args.get('page', 1, type=int)
    per_page = 25

    if request.method == 'POST':
        return views.delete_selected_users() # Call the dedicated delete function

    users = User.query.paginate(page=page, per_page=per_page, error_out=False)
    return render_template('show_users.html', users=users, pages=users.pages,
                           next_url=users.next_num if users.has_next else None,
                           prev_url=users.prev_num if users.has_prev else None,
                           current_page=users.page)


# --- Update dashboard route ---
@app.route("/dashboard", methods=["GET", "POST"])
@login_required
def dashboard():
    form = UpdatePasswordForm()
    # Original: user = User.query.get(current_user.id)
    # New:
    user = db.session.get(User, current_user.id) # Use db.session.get

    if user and user.favorites:
        # Handle potential JSON string or already decoded list/dict
        if isinstance(user.favorites, str):
            try:
                favorites_list = json.loads(user.favorites)
            except json.JSONDecodeError:
                favorites_list = [] # Handle invalid JSON
        else:
            favorites_list = user.favorites if user.favorites is not None else []
    else:
        favorites_list = []

    return render_template("dashboard.html", favorites=favorites_list, form=form)
# --- End of update ---

@login_manager.unauthorized_handler
def unauthorized_callback():
    return redirect(url_for("login"))  #


# --- Update load_user function ---
@login_manager.user_loader
def load_user(user_id):
    # Original: return User.query.get(int(user_id))
    # New:
    return db.session.get(User, int(user_id))
# --- End of update ---


if __name__ == "__main__":
    app.run(debug=False, use_reloader=True)