import os
import sys
from ai import call_AI
from models import Call
import json
from utils import get_call_by_id
import logging


def setup_logger(log_filename):
    """Sets up the AI logger with proper error handling."""
    try:
        logs_dir="logs"
        log_path = os.path.join(os.path.dirname(__file__), logs_dir) # Correct path
        os.makedirs(log_path, exist_ok=True)

        log_file_path = os.path.join(log_path, log_filename)
        logger = logging.getLogger('ai_processor')  # Standard logger name
        logger.setLevel(logging.INFO)  # Set level for the logger

        if not logger.handlers:  # Check if handler already exists
            handler = logging.FileHandler(log_file_path, encoding='utf-8')
            handler.setLevel(logging.INFO) 
            formatter = logging.Formatter('%(asctime)s - %(levelname)s - %(filename)s - %(message)s')
            handler.setFormatter(formatter)
            logger.addHandler(handler)

    except Exception as e:  # Catch broader exceptions during setup
        print(f"Error setting up logging: {e}", file=sys.stderr) # Important for debugging
        return None # Indicate setup failure

    return logger


# Call the setup_logging function at the top level 
ai_logger = setup_logger('ai_processing.log') # Pass in log filename



def create_summary(db, call_id):
    """Generates and saves a summary for a call if one doesn't already exist."""
    try:
        call = get_call_by_id(db, call_id)
        if not call:
            ai_logger.error(f"Call with ID {call_id} not found.")
            return None

        if not call.summary:
            json_transcript = call.json_transcript
            if json_transcript is None:
                ai_logger.warning(f"Transcript data missing for Call ID {call_id}.")
                return None

            if not isinstance(json_transcript, str):
                try:
                    json_transcript = json.dumps(json_transcript, indent=2)
                except (TypeError, json.JSONDecodeError) as e:
                    ai_logger.error(f"Error converting transcript to JSON for Call ID {call_id}: {e}")
                    return None

            summary = generate_summary(json_transcript)

            if summary is None:
                ai_logger.error(f"Could not generate summary for Call ID {call_id}. Check AI API call.")
                return None

            # Cleaning summary remains the same
            summary = summary.replace("```", "").strip()
            if summary.startswith("html"):
                summary = summary[4:].strip()

            call.summary = summary
            db.session.commit()
            ai_logger.info(f"Summary generated for Call ID {call_id}")  # Changed to info
        else:
            ai_logger.info(f"Summary already exists for Call ID {call_id}") # Changed to info
        return call

    except Exception as e:
        ai_logger.exception(f"An unexpected error occurred during summary creation for Call ID {call_id}: {e}") # Changed to exception to log traceback
        db.session.rollback()
        return None


def get_prompt_from_file(filename):
    """Reads and returns the prompt from a file in the /prompts directory."""
    prompts_dir = "prompts"
    filepath = os.path.join(prompts_dir, filename)

    if not os.path.exists(filepath):  # Check directly for the file's existence
        ai_logger.error(f"Prompt file not found: {filepath}") # More specific logging
        return None

    try:
        with open(filepath, "r", encoding="utf-8") as f:
            prompt = f.read()
            return prompt
    except Exception as e: # Catching all exceptions here is fine for this use case, but usually, it's better to catch specific exceptions
        ai_logger.exception(f"An error occurred while reading the prompt file: {e}")
        return None
    

def create_complete_prompt(prompt, transcript):
    """Creates a prompt with the transcript enclosed in a JSON code block."""
    return f"{prompt}```json Transcript:\n{transcript}\n```"


def generate_summary(json_transcript):
    """Generates a summary using the OpenAI API."""
    prompt = get_prompt_from_file("summary_prompt.txt")
    if prompt is None:
        return None  # Handle error from get_prompt_from_file

    complete_prompt = create_complete_prompt(prompt, json_transcript)
    try:
        summary = call_AI(complete_prompt) # Assuming Call_AI handles its own errors. This just returns.
        return summary
    except Exception as e:  # Catch potential exceptions during API call
        ai_logger.exception(f"Error generating summary: {e}")
        return None