import os
import sys
from models import  User, db
from itsdangerous import SignatureExpired
from flask import current_app, url_for, flash, redirect
import smtplib
from email.message import EmailMessage
import logging
from datetime import datetime


def setup_logger(log_filename):
    """Sets up the email logger with proper error handling."""
    try:
        logs_dir="logs"
        log_path = os.path.join(os.path.dirname(__file__), logs_dir) # Correct path
        os.makedirs(log_path, exist_ok=True)

        log_file_path = os.path.join(log_path, log_filename)
        logger = logging.getLogger('email_processor')  # Standard logger name
        logger.setLevel(logging.INFO)  # Set level for the logger

        if not logger.handlers:  # Check if handler already exists
            handler = logging.FileHandler(log_file_path, encoding='utf-8')
            handler.setLevel(logging.INFO) 
            formatter = logging.Formatter('%(asctime)s - %(levelname)s - %(filename)s - %(message)s')
            handler.setFormatter(formatter)
            logger.addHandler(handler)

    except Exception as e:  # Catch broader exceptions during setup
        logger.exception(f"Error setting up logging: {e}", file=sys.stderr) # Important for debugging
        return None # Indicate setup failure

    return logger


# Call the setup_logging function at the top level 
email_logger = setup_logger('email_processing.log') # Pass in log filename


def log_email(user_id, username, to_address, symbol, call_id, success=True, error_message=None):
    if success:
        log_message = f"Email sent successfully to {to_address} for user {username} (ID: {user_id}), symbol: {symbol}, call ID: {call_id}"
    else:
        log_message = f"Failed to send email to {to_address} for user {username} (ID: {user_id}), symbol: {symbol}, call ID: {call_id}. Error: {error_message}"

    email_logger.info(log_message) 


def send_new_transcripts_email(db, matching_favorites):
    """Sends email notifications to users about new transcripts for their favorite symbols."""

    with current_app.app_context():
        for user_id, username, symbol, call_id in matching_favorites:
            user = db.session.query(User).get(user_id)
            if user:
                subject = f"New Transcript Available for {symbol}!"
                call_link = url_for('call_page', call_id=call_id, _external=True)
                body = (
                    f"Hi {username},\n\n"
                    f"A new transcript is available for {symbol}, one of your favorite companies.\n\n"
                    f"Click here to read it: {call_link}\n\n"
                    f"Best,\n"
                    f"The Sandia Team"
                )
                try:
                    send_email(user.email, subject, body)
                    log_email(user_id, username, user.email, symbol, call_id, success=True) 
                except Exception as e:
                    log_email(user_id, username, user.email, symbol, call_id, success=False, error_message=str(e)) 


def resend_verification_email(email):
    from app import s
    subject = "Welcome to Sandia. Verify your email for 6 months FREE"
    token = s.dumps(email, salt="email-confirm")
    body = (
        f"Thanks for signing up for Sandia!<br>We're currently in Beta, so expect a few bumps along the way. To show our appreciation, we're giving you 6 months of FREE access!<br>"
        f"Click here to verify your email and get started:<br>"
        f'{url_for("confirm_email", token=token, _external=True)}<br>'
        f"(Didn't sign up? No worries, just ignore this email.)<br>"
        f"Cheers,<br>The Sandia Team"
    )
    send_email(email, subject, body)


def send_verification_email(username, email):
    from app import s
    subject = "Welcome to Sandia. Verify your email for 6 months FREE"
    token = s.dumps(email, salt="email-confirm")
    body = (
        f"Hey {username},<br>"
        f"Thanks for signing up for Sandia!<br>We're currently in Beta, so expect a few bumps along the way.  To show our appreciation, we're giving you 6 months of FREE access!<br>"
        f"Click here to verify your email and get started:<br>"
        f'{url_for("confirm_email", token=token, _external=True)}<br>'
        f"(Didn't sign up? No worries, just ignore this email.)<br>"
        f"Cheers,<br>The Sandia Team"
    )
    send_email(email, subject, body)
    log_email(None, username, email, None, None, success=True) 
    # send_email("info@sandia.com", subject, body)


def send_email(to_address, subject, body):
    from app import SMTP_PASSWORD
    from_address = "info@sandia.com"
    login_name = "598624004@smtp-brevo.com"
    servername = "smtp-relay.brevo.com"
    port = 587
    password = SMTP_PASSWORD
    msg = EmailMessage()
    msg["From"] = from_address
    msg["To"] = to_address
    msg["Subject"] = subject
    msg.set_content(body)

    try:
        with smtplib.SMTP(servername, port) as server:  # 'with' statement handles closing
            server.starttls()
            server.login(login_name, password)
            server.sendmail(from_address, to_address, msg.as_string())

    except smtplib.SMTPException as e:
        email_logger.exception(f"Failed to send email: {e}")  # Log the exception with traceback
    except Exception as e:
        email_logger.exception(f"An unexpected error occurred: {e}")  # Log the exception with traceback



def confirm_email(token):
    from app import s
    try:
        email = s.loads(token, salt="email-confirm", max_age=86400)
    except SignatureExpired:
        flash("The confirmation link has expired. Please register again.", "danger")
        return redirect(url_for("register"))

    user = User.query.filter_by(email=email).first()
    if not user:
        flash("Invalid confirmation link.", "danger")
        return redirect(url_for("register"))

    user.confirmed = True
    db.session.commit()
    flash("Your email has been confirmed. You can now log in.", "success")
    return redirect(url_for("login"))