from flask_wtf import FlaskForm
# --- START OF CHANGES ---
from wtforms import (
    DateTimeField, # Use DateTimeField instead of DateTimeLocalField
    IntegerField, StringField, PasswordField, SubmitField, BooleanField, TextAreaField,
    SelectField, DecimalField, FloatField
)
# Import Optional validator
from wtforms.validators import DataRequired, Length, Email, EqualTo, NumberRange, Optional
# --- END OF CHANGES ---

class RegistrationForm(FlaskForm):
    username = StringField('Username',validators=[DataRequired(), Length(min=2,max=20)])
    email = StringField('Email',validators=[DataRequired(), Email()])
    password = PasswordField('Password', validators=[DataRequired()])
    confirm_password = PasswordField('Confirm Password', validators=[DataRequired(), EqualTo('password')])
    submit = SubmitField('Sign Up')


class UpdateUserForm(FlaskForm):
    username = StringField('Username',validators=[DataRequired(), Length(min=2,max=20)])
    email = StringField('Email',validators=[DataRequired(), Email()])
   # --- START OF CHANGES ---
    # Change 'No' value to '0' for easier boolean conversion
    email_confirmed = SelectField('Email Confirmed', choices=[('1', 'Yes'), ('0', 'No')], default='0')
    # Add Optional() and NumberRange validators
    login_count = IntegerField('Login Count', validators=[Optional(), NumberRange(min=0)])
    # Use DateTimeField and add Optional() validator
    last_login = DateTimeField(
        'Last Login',
        format='%Y-%m-%dT%H:%M', # Format expected from datetime-local input
        validators=[Optional()]
    )
    # Removed favorites field as it's not handled in the update logic
    # favorites = StringField('Favorite IDs (comma-separated)')
    # --- END OF CHANGES ---
    submit = SubmitField('Update')


class LoginForm(FlaskForm):
    username = StringField('Username',validators=[DataRequired()])
    password = PasswordField('Password', validators=[DataRequired()])
    remember = BooleanField('Remember Me')
    submit = SubmitField('Login')


class ReverifyEmailForm(FlaskForm):
    email = StringField('Email',validators=[DataRequired(), Email()])
    submit = SubmitField('Reverify Email')


class ResetPasswordForm(FlaskForm):
    email = StringField('Email',validators=[DataRequired(), Email()])
    submit = SubmitField('Reset')


class UpdatePasswordForm(FlaskForm):
    current_password = PasswordField('Current Password', validators=[DataRequired()])
    new_password = PasswordField('Password', validators=[DataRequired()])
    confirm_password = PasswordField('Confirm Password', validators=[DataRequired(), EqualTo('new_password')])
    submit = SubmitField('Update Password')


class OptionsForm(FlaskForm):
    model = SelectField('Model', choices=[('1', 'gpt-3.5-turbo'), ('2', 'gpt-4')], default='2') 
    temperature = DecimalField('Temperature', validators=[NumberRange(min=0.1, max=1.0)]) 
    submit = SubmitField('Enter')


class CreateCallForm(FlaskForm):
    title = StringField('Title', validators=[DataRequired(), Length(min=1, max=250)])
    text = TextAreaField('Text', validators=[DataRequired(), Length(min=1, max=50000)])
    summary = TextAreaField('Summary', validators=[Length(min=0, max=50000)])
    companyname = StringField('Company Name', validators=[DataRequired(), Length(min=1, max=250)])
    symbol = StringField('Symbol', validators=[DataRequired(), Length(min=1, max=10)])
    year = StringField('Year', validators=[DataRequired()])
    quarter = StringField('Quarter', validators=[DataRequired()])
    submit = SubmitField('Create')


class EditCallForm(FlaskForm):
    title = StringField('Title', validators=[DataRequired(), Length(min=1, max=250)])
    json_transcript = TextAreaField('Text', validators=[DataRequired(), Length(min=1, max=50000)])
    summary = TextAreaField('Summary', validators=[Length(min=0, max=50000)])
    companyname = StringField('Company Name', validators=[DataRequired(), Length(min=1, max=250)])
    symbol = StringField('Symbol', validators=[DataRequired(), Length(min=1, max=10)])
    year = StringField('Year', validators=[DataRequired()])
    quarter = StringField('Quarter', validators=[DataRequired()])
    submit = SubmitField('Save')